<?php

namespace App\Http\Controllers;

use App\Models\ViewCompany;
use App\Models\Company;
use App\Models\Complaint;
use App\Models\DeleteRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class ViewCompanyController extends Controller
{
        // Show all companies with their complaints
        public function viewCompaniesWithComplaints()
        {
            $companies = Company::with('complaints')->get();
            return view('company.index', compact('companies'));
        }

        // Delete a specific company and its complaints
        public function deleteCompany($id)
        {
            $company = Company::findOrFail($id);
            $company->complaints()->delete(); // delete related complaints
            $company->delete(); // delete company
            return redirect()->back()->with('success', 'Company and complaints deleted successfully.');
        }

        // Delete a single complaint
        public function deleteComplaint($id)
        {
            $complaint = Complaint::findOrFail($id);
            $complaint->delete();
            return redirect()->back()->with('success', 'Complaint deleted successfully.');
        }


    /**
     * Show admin dashboard with statistics
     */
    public function dashboard()
    {
        $stats = [
            'total_companies' => Company::count(),
            'total_complaints' => Complaint::count(),
            'pending_complaints' => Complaint::where('status', 'pending')->count(),
            'delete_requests' => DeleteRequest::where('status', 'pending')->count()
        ];

        return view('admin.auth.dashboard', compact('stats'));
    }

    /**
     * Show admin profile
     */
    public function showProfile()
    {
        $admin = Auth::guard('admin')->user();
        return view('admin.profile.show', compact('admin'));
    }

    /**
     * Update admin profile
     */
    public function updateProfile(Request $request)
    {
        $admin = Auth::guard('admin')->user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:admins,email,'.$admin->id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        // Handle avatar upload
        if ($request->hasFile('avatar')) {
            // Delete old avatar if exists
            if ($admin->avatar) {
                Storage::delete('public/'.$admin->avatar);
            }

            $path = $request->file('avatar')->store('admin/avatars', 'public');
            $validated['avatar'] = $path;
        }

        $admin->update($validated);

        return redirect()->route('admin.profile')->with('success', 'Profile updated successfully');
    }

    /**
     * List all companies
     */
    public function companies()
    {
        $companies = Company::withCount('complaints')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('companies.index', compact('companies'));
    }


    /**
     * List all complaints
     */
    public function complaints()
    {
        $complaints = Complaint::with(['company', 'employee'])
            ->filter(request(['status', 'search']))
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('admin.complaints.index', compact('complaints'));
    }

    /**
     * Show single complaint
     */
    public function showComplaint(Complaint $complaint)
    {
        // Generate secure URL for termination letter
        $terminationLetterUrl = Storage::temporaryUrl(
            $complaint->termination_letter_path,
            now()->addMinutes(30)
        );

        return view('company.complaints.show', [
            'complaint' => $complaint,
            'terminationLetterUrl' => $terminationLetterUrl
        ]);
    }

    /**
     * Update complaint status
     */
    public function updateComplaintStatus(Request $request, Complaint $complaint)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,under_investigation,resolved,rejected'
        ]);

        $complaint->update($validated);

        return back()->with('success', 'Complaint status updated successfully');
    }

    /**
     * List delete requests
     */
    public function deleteRequests()
    {
        $requests = DeleteRequest::with(['employee', 'payment'])
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('admin.delete-requests.index', compact('requests'));
    }

    /**
     * Approve delete request
     */
    public function approveDeleteRequest(DeleteRequest $deleteRequest)
    {
        // Delete employee records (implement soft delete if needed)
        $deleteRequest->employee()->delete();

        // Update request status
        $deleteRequest->update(['status' => 'approved']);

        return back()->with('success', 'Delete request approved successfully');
    }

    /**
     * Reject delete request
     */
    public function rejectDeleteRequest(DeleteRequest $deleteRequest)
    {
        $deleteRequest->update(['status' => 'rejected']);
        return back()->with('success', 'Delete request rejected');
    }
}
