<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Complaint;
use App\Models\Employee;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ComplaintController extends Controller
{
    public function create()
    {
        // return view('company.complaint_form');
        return view('frontend.pages.company.complaints.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|integer', // Allow manual entry of Employee ID
            'employee_name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'surname' => 'required|string|max:255',
            'aadhar_id' => 'required|digits:12', // Aadhar can be repeated
            'reason' => 'required|string|max:1000',
            'termination_letter' => 'nullable|file|mimes:pdf|max:2048'
        ]);

        // Ensure a company is logged in
        $company_id = auth()->guard('company')->id();
        if (!$company_id) {
            return redirect()->back()->withErrors(['error' => 'Unauthorized access. Please log in.']);
        }

        // Store termination letter (if uploaded)
        $path = null;
        if ($request->hasFile('termination_letter')) {
            $path = $request->file('termination_letter')->store('termination-letters', 'public'); // Store in 'public' if needed
        }

        // Create complaint entry
        Complaint::create([
            'company_id' => $company_id,
            'employee_id' => $validated['employee_id'],
            'employee_name' => $validated['employee_name'],
            'father_name' => $validated['father_name'],
            'surname' => $validated['surname'],
            'aadhar_id' => $validated['aadhar_id'],
            'reason' => $validated['reason'],
            'termination_letter_path' => $path
        ]);

        return redirect()->route('company.dashboard')
            ->with('success', 'Complaint registered successfully');
    }

    public function search(Request $request)
    {
        $request->validate([
            'employee_name' => 'required|string',
            'father_name' => 'required|string',
            'surname' => 'required|string',
        ]);

        $employee_name = $request->input('employee_name');
        $father_name = $request->input('father_name');
        $surname = $request->input('surname');

        // Employee search query
        $complaints = Complaint::where('employee_name', 'LIKE', "%$employee_name%")
                                ->where('father_name', 'LIKE', "%$father_name%")
                                ->where('surname', 'LIKE', "%$surname%")
                                ->get();

        if ($complaints->isEmpty()) {
            // return back()->with('error', 'No complaints found for this employee.');
            return view('frontend.pages.company.complaints.search', [
                'complaints' => $complaints,
                'error' => 'No past complaints registered against this employee.'
            ]);
        }

        return view('frontend.pages.company.complaints.search', compact('complaints'));
    }

    public function destroy($id)
    {
        $complaint = Complaint::findOrFail($id);
        $complaint->delete();

        return redirect()->back()->with('success', 'Complaint deleted successfully.');
    }

    public function edit($id)
    {
        // This fetches a single complaint record by its ID
        $complaint = Complaint::findOrFail($id);

        // Pass the single complaint to the view
        return view('Frontend.pages.Company.complaints.edit', compact('complaint'));
    }


    // Handle Update

    public function update(Request $request, $id)
        {
            $complaint = Complaint::findOrFail($id);

            $validated = $request->validate([
                'employee_name' => 'required|string|max:255',
                'father_name' => 'required|string|max:255',
                'surname' => 'required|string|max:255',
                'aadhar_id' => 'required|string|max:255',
                'reason' => 'required|string',
                'termination_letter' => 'nullable|mimes:pdf|max:2048',
            ]);

            // Update complaint data
            // $complaint->update($validated);


            if ($request->hasFile('termination_letter')) {
                $path = $request->file('termination_letter')->store('termination_letters', 'public');
                $complaint->update(['termination_letter_path' => $path]);
            }

            // Update complaint data
            $complaint->update($request->all());

            // Redirect to the complaints index or back to the same page with success message
            return redirect()->route('companies.list')->with('success', 'Complaint updated successfully');

            // return redirect()->route('complaints.edit', $complaint->id)
            //                  ->with('success', 'Complaint updated successfully.');
    }

    


}
