<?php

namespace App\Http\Controllers;

use App\Models\Company;
use Illuminate\Http\Request;
use App\Models\Complaint;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class CompanyController extends Controller
{

    public function dashboard()
    {
        // Assuming 'company' guard is used for company authentication
        $companyId = Auth::guard('company')->id();

        // Fetch only complaints associated with the authenticated company
        $complaints = Complaint::where('company_id', $companyId)->get();

        //To fetch Comapny name
        $user = Auth::guard('company')->user();


        return view('frontend.pages.company.dashboard', compact('complaints'));
    }

    // Show all companies (Admin)
    // public function index()
    // {
    //     $companies = Company::paginate(10);
    //     return view('admin.pages.CompanyInfo.Company.index', compact('companies'));
    // }


    public function index(Request $request)
{
    $search = $request->input('search');

    $companies = Company::when($search, function ($query, $search) {
        $query->where('name', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%")
              ->orWhere('pancard', 'like', "%{$search}%")
              ->orWhere('contact', 'like', "%{$search}%")
              ->orWhere('registration_number', 'like', "%{$search}%");
    })->paginate(10);

    return view('admin.pages.CompanyInfo.Company.index', compact('companies'));
}

    // Show create form (Admin)
    public function create()
    {
        return view('admin.pages.CompanyInfo.Company.create');
    }

    // Store new company (Admin)
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:companies',
            'password' => 'required|min:8|confirmed',
            'registration_number' => 'required|unique:companies',
            'pancard' => 'required|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/',
            'contact' => 'required|string|max:15',
            'address' => 'required|string',
        ]);

        $validated['password'] = Hash::make($validated['password']);

        Company::create($validated);

        return redirect()->route('admin.company.index')
            ->with('success', 'Company created successfully');
    }

    // Show edit form (Admin)
    public function edit(Company $company)
    {
        return view('admin.pages.CompanyInfo.Company.edit', compact('company'));
    }

    // Update company (Admin)
    public function update(Request $request, Company $company)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:companies,email,'.$company->id,
            'password' => 'sometimes|nullable|min:8|confirmed',
            'registration_number' => 'required|unique:companies,registration_number,'.$company->id,
            'pancard' => 'required|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/',
            'contact' => 'required|string|max:15',
            'address' => 'required|string',
        ]);

        if (!empty($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        } else {
            unset($validated['password']);
        }

        $company->update($validated);

        return redirect()->route('admin.company.index')
            ->with('success', 'Company updated successfully');
    }

    // Delete company (Admin)
    public function destroy(Company $company)
    {
        $company->complaints()->delete();
        $company->delete();

        return redirect()->route('admin.company.index')
            ->with('success', 'Company deleted successfully');
    }

    // Show company details (Admin)
    public function show(Company $company)
    {
        $complaints = $company->complaints()->paginate(10);
        return view('admin.pages.CompanyInfo.Company.show', compact('company', 'complaints'));
    }

    public function showAllCompanies()
    {
        // Ensure that only admins can view all companies
        // if (Auth::user()->role !== 'admin') {
        //     return redirect()->route('home')->with('error', 'Access denied');
        // }

        // Fetch all companies
        // $companies = Company::all();
        $companies = Company::paginate(10);

        return view('frontend.pages.Company.complaints.index', compact('companies'));
    }


    public function showAllComplaints()
    {
        // Ensure that only admins can view all companies
        // if (Auth::user()->role !== 'admin') {
        //     return redirect()->route('home')->with('error', 'Access denied');
        // }

        // Fetch all companies
        // $companies = Company::all();
        $companies = Company::paginate(10);

        return view('admin.pages.CompanyInfo.complaints.index', compact('companies'));
    }
}

// namespace App\Http\Controllers;

// use App\Models\Company;
// use Illuminate\Http\Request;
// use App\Models\Complaint;
// use Illuminate\Support\Facades\Auth;

// class CompanyController extends Controller
// {
//     /**
//      * Dashboard - Shows complaints of the authenticated company
//      */
//     public function dashboard()
//     {
//         // Assuming 'company' guard is used for company authentication
//         $companyId = Auth::guard('company')->id();

//         // Fetch only complaints associated with the authenticated company
//         $complaints = Complaint::where('company_id', $companyId)->get();

//         return view('frontend.pages.company.dashboard', compact('complaints'));
//     }

//     /**
//      * List all companies (admin only)
//      */
//     public function index()
//     {
//         // Ensure that only admins can view the list of all companies
//         if (Auth::user()->role !== 'admin') {
//             return redirect()->route('home')->with('error', 'Access denied');
//         }

//         // Fetch all companies along with their complaints using eager loading
//         $companies = Company::with('complaints')->get();

//         return view('company.index', compact('companies'));
//     }

//     /**
//      * Show a list of all companies for admin (can be accessed only by admins)
//      */
//     public function showAllCompanies()
//     {
//         // Ensure that only admins can view all companies
//         // if (Auth::user()->role !== 'admin') {
//         //     return redirect()->route('home')->with('error', 'Access denied');
//         // }

//         // Fetch all companies
//         // $companies = Company::all();
//         $companies = Company::paginate(10);

//         return view('frontend.pages.Company.complaints.index', compact('companies'));
//     }

//     /**
//      * Show the form to create a new company
//      */
//     public function create()
//     {
//         return view('admin.pages.CompanyInfo.Company.create'); // Add your create form here
//     }

//     /**
//      * Store a new company in the database
//      */
//     public function store(Request $request)
//     {
//         // Validate the incoming request
//         $request->validate([
//             'name' => 'required|string|max:255',
//             'email' => 'required|string|email|max:255|unique:companies',
//             'password' => 'required|string|min:6|confirmed',
//             'registration_number' => 'required|string|unique:companies',
//             'address' => 'required|string',
//             'pancard' => 'required|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/',
//             'contact' => 'required|string|max:15',
//         ]);

//         // Create a new company
//         Company::create($request->all());

//         return redirect()->route('admin.pages.CompanyInfo.Company.create')->with('success', 'Company created successfully!');
//     }

//     /**
//      * Show the form to edit a company's details
//      */
//     public function edit(Company $company)
//     {
//         return view('admin.pages.CompanyInfo.Company.edit', compact('company')); // Add your edit form here
//     }

//     /**
//      * Update the company's information
//      */
//     public function update(Request $request, Company $company)
//     {
//         // Validate the incoming request
//         $request->validate([
//             'name' => 'required|string|max:255',
//             'email' => 'required|string|email|max:255|unique:companies',
//             'password' => 'required|string|min:6|confirmed',
//             'registration_number' => 'required|string|unique:companies',
//             'address' => 'required|string',
//             'pancard' => 'required|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/',
//             'contact' => 'required|string|max:15',
//         ]);

//         // Update the company details
//         $company->update($request->all());

//         return redirect()->route('admin.pages.CompanyInfo.Company.index')->with('success', 'Company updated successfully!');
//     }

//     /**
//      * Delete a company (with its associated complaints)
//      */
//     public function destroy(Company $company)
//     {
//         // Optionally, delete all complaints associated with this company before deleting the company
//         $company->complaints()->delete();
//         $company->delete();

//         return redirect()->route('admin.pages.CompanyInfo.Company.index')->with('success', 'Company deleted successfully!');
//     }

//     /**
//      * Show all complaints for a specific company (if needed)
//      */
//     public function showComplaints(Company $company)
//     {
//         // Show the complaints of a specific company
//         $complaints = $company->complaints;
//         return view('admin.pages.CompanyInfo.Company.show', compact('company', 'complaints'));
//     }

// }











// namespace App\Http\Controllers;

// use App\Models\Company;
// use Illuminate\Http\Request;
// use App\Models\Complaint;
// use Illuminate\Support\Facades\Auth;
//  // Add this at the top


// class CompanyController extends Controller
// {

//     public function dashboard()
//     {
//         $companyId = Auth::guard('company')->id();

//         // सिर्फ उसी कंपनी की शिकायतें प्राप्त करें
//         $complaints = Complaint::where('company_id', $companyId)->get();

//         return view('frontend.pages.company.dashboard', compact('complaints'));
//     }

//     // List Companies (if needed)
//     public function index()
//     {

//         // Fetch all companies with their associated complaints using eager loading

//         // $companies = Company::all();
//         $companies = []; // Add your logic
//         return view('company.index', compact('companies'));
//     }


//     public function showAllCompanies()
//     {
//         $companies = Company::all(); // Fetch all companies
//         return view('Frontend.pages.Company.complaints.index', compact('companies'));
//     }

// }

